/**
  ******************************************************************************
  * @file    USB_Device/DualCore_Standalone/Src/usbd_cdc_desc.c
  * @author  MCD Application Team
  * @brief   This file provides the USBD descriptors and string formatting method.
  ******************************************************************************
  * @attention
  *
  * Copyright (c) 2017 STMicroelectronics.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */
/* Includes ------------------------------------------------------------------ */
#include "usbd_core.h"
#include "usbd_desc.h"
#include "usbd_conf.h"

/* Private typedef ----------------------------------------------------------- */
/* Private define ------------------------------------------------------------ */
#define USBD_VCP_VID                      0x0483
#define USBD_VCP_PID                      0x5740
#define USBD_VCP_LANGID_STRING            0x409
#define USBD_VCP_MANUFACTURER_STRING      "STMicroelectronics"
#define USBD_VCP_PRODUCT_HS_STRING        "STM32 Virtual ComPort in HS Mode"
#define USBD_VCP_SERIALNUMBER_HS_STRING   "00000000001A"
#define USBD_VCP_PRODUCT_FS_STRING        "STM32 Virtual ComPort in FS Mode"
#define USBD_VCP_SERIALNUMBER_FS_STRING   "00000000001B"
#define USBD_VCP_CONFIGURATION_HS_STRING  "VCP Config"
#define USBD_VCP_INTERFACE_HS_STRING      "VCP Interface"
#define USBD_VCP_CONFIGURATION_FS_STRING  "VCP Config"
#define USBD_VCP_INTERFACE_FS_STRING      "VCP Interface"

/* Private macro ------------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
uint8_t *USBD_VCP_DeviceDescriptor(USBD_SpeedTypeDef speed, uint16_t * length);
uint8_t *USBD_VCP_LangIDStrDescriptor(USBD_SpeedTypeDef speed,
                                      uint16_t * length);
uint8_t *USBD_VCP_ManufacturerStrDescriptor(USBD_SpeedTypeDef speed,
                                            uint16_t * length);
uint8_t *USBD_VCP_ProductStrDescriptor(USBD_SpeedTypeDef speed,
                                       uint16_t * length);
uint8_t *USBD_VCP_SerialStrDescriptor(USBD_SpeedTypeDef speed,
                                      uint16_t * length);
uint8_t *USBD_VCP_ConfigStrDescriptor(USBD_SpeedTypeDef speed,
                                      uint16_t * length);
uint8_t *USBD_VCP_InterfaceStrDescriptor(USBD_SpeedTypeDef speed,
                                         uint16_t * length);

/* Private variables --------------------------------------------------------- */
USBD_DescriptorsTypeDef VCP_Desc = {
  USBD_VCP_DeviceDescriptor,
  USBD_VCP_LangIDStrDescriptor,
  USBD_VCP_ManufacturerStrDescriptor,
  USBD_VCP_ProductStrDescriptor,
  USBD_VCP_SerialStrDescriptor,
  USBD_VCP_ConfigStrDescriptor,
  USBD_VCP_InterfaceStrDescriptor,
};

/* USB Standard Device Descriptor */
#if defined ( __ICCARM__ )      /* !< IAR Compiler */
#pragma data_alignment=4
#endif
__ALIGN_BEGIN uint8_t USBD_VCP_DeviceDesc[USB_LEN_DEV_DESC] __ALIGN_END = {
  0x12,                         /* bLength */
  USB_DESC_TYPE_DEVICE,         /* bDescriptorType */
  0x00,                         /* bcdUSB */
  0x02,
  0x02,                         /* bDeviceClass */
  0x02,                         /* bDeviceSubClass */
  0x00,                         /* bDeviceProtocol */
  USB_MAX_EP0_SIZE,             /* bMaxPacketSize */
  LOBYTE(USBD_VCP_VID),         /* idVendor */
  HIBYTE(USBD_VCP_VID),         /* idVendor */
  LOBYTE(USBD_VCP_PID),         /* idVendor */
  HIBYTE(USBD_VCP_PID),         /* idVendor */
  0x00,                         /* bcdDevice rel. 2.00 */
  0x02,
  USBD_IDX_MFC_STR,             /* Index of manufacturer string */
  USBD_IDX_PRODUCT_STR,         /* Index of product string */
  USBD_IDX_SERIAL_STR,          /* Index of serial number string */
  USBD_MAX_NUM_CONFIGURATION    /* bNumConfigurations */
};                              /* USB_DeviceDescriptor */

/* USB Standard Device Descriptor */
#if defined ( __ICCARM__ )      /* !< IAR Compiler */
#pragma data_alignment=4
#endif
__ALIGN_BEGIN uint8_t USBD_VCP_LangIDDesc[USB_LEN_LANGID_STR_DESC] __ALIGN_END = {
  USB_LEN_LANGID_STR_DESC,
  USB_DESC_TYPE_STRING,
  LOBYTE(USBD_VCP_LANGID_STRING),
  HIBYTE(USBD_VCP_LANGID_STRING),
};

#if defined ( __ICCARM__ )      /* !< IAR Compiler */
#pragma data_alignment=4
#endif
__ALIGN_BEGIN uint8_t USBD_VCP_StrDesc[USBD_MAX_STR_DESC_SIZ] __ALIGN_END;

/* Private functions --------------------------------------------------------- */

/**
  * @brief  Returns the device descriptor. 
  * @param  speed: Current device speed
  * @param  length: Pointer to data length variable
  * @retval Pointer to descriptor buffer
  */
uint8_t *USBD_VCP_DeviceDescriptor(USBD_SpeedTypeDef speed, uint16_t * length)
{
  *length = sizeof(USBD_VCP_DeviceDesc);
  return USBD_VCP_DeviceDesc;
}

/**
  * @brief  Returns the LangID string descriptor.        
  * @param  speed: Current device speed
  * @param  length: Pointer to data length variable
  * @retval Pointer to descriptor buffer
  */
uint8_t *USBD_VCP_LangIDStrDescriptor(USBD_SpeedTypeDef speed,
                                      uint16_t * length)
{
  *length = sizeof(USBD_VCP_LangIDDesc);
  return USBD_VCP_LangIDDesc;
}

/**
  * @brief  Returns the product string descriptor. 
  * @param  speed: Current device speed
  * @param  length: Pointer to data length variable
  * @retval Pointer to descriptor buffer
  */
uint8_t *USBD_VCP_ProductStrDescriptor(USBD_SpeedTypeDef speed,
                                       uint16_t * length)
{
  if (speed == 0)
  {
    USBD_GetString((uint8_t *) USBD_VCP_PRODUCT_HS_STRING, USBD_VCP_StrDesc,
                   length);
  }
  else
  {
    USBD_GetString((uint8_t *) USBD_VCP_PRODUCT_FS_STRING, USBD_VCP_StrDesc,
                   length);
  }
  return USBD_VCP_StrDesc;
}

/**
  * @brief  Returns the manufacturer string descriptor. 
  * @param  speed: Current device speed
  * @param  length: Pointer to data length variable
  * @retval Pointer to descriptor buffer
  */
uint8_t *USBD_VCP_ManufacturerStrDescriptor(USBD_SpeedTypeDef speed,
                                            uint16_t * length)
{
  USBD_GetString((uint8_t *) USBD_VCP_MANUFACTURER_STRING, USBD_VCP_StrDesc,
                 length);
  return USBD_VCP_StrDesc;
}

/**
  * @brief  Returns the serial number string descriptor.        
  * @param  speed: Current device speed
  * @param  length: Pointer to data length variable
  * @retval Pointer to descriptor buffer
  */
uint8_t *USBD_VCP_SerialStrDescriptor(USBD_SpeedTypeDef speed,
                                      uint16_t * length)
{
  if (speed == USBD_SPEED_HIGH)
  {
    USBD_GetString((uint8_t *) USBD_VCP_SERIALNUMBER_HS_STRING,
                   USBD_VCP_StrDesc, length);
  }
  else
  {
    USBD_GetString((uint8_t *) USBD_VCP_SERIALNUMBER_FS_STRING,
                   USBD_VCP_StrDesc, length);
  }
  return USBD_VCP_StrDesc;
}

/**
  * @brief  Returns the configuration string descriptor.    
  * @param  speed: Current device speed
  * @param  length: Pointer to data length variable
  * @retval Pointer to descriptor buffer
  */
uint8_t *USBD_VCP_ConfigStrDescriptor(USBD_SpeedTypeDef speed,
                                      uint16_t * length)
{
  if (speed == USBD_SPEED_HIGH)
  {
    USBD_GetString((uint8_t *) USBD_VCP_CONFIGURATION_HS_STRING,
                   USBD_VCP_StrDesc, length);
  }
  else
  {
    USBD_GetString((uint8_t *) USBD_VCP_CONFIGURATION_FS_STRING,
                   USBD_VCP_StrDesc, length);
  }
  return USBD_VCP_StrDesc;
}

/**
  * @brief  Returns the interface string descriptor.        
  * @param  speed: Current device speed
  * @param  length: Pointer to data length variable
  * @retval Pointer to descriptor buffer
  */
uint8_t *USBD_VCP_InterfaceStrDescriptor(USBD_SpeedTypeDef speed,
                                         uint16_t * length)
{
  if (speed == 0)
  {
    USBD_GetString((uint8_t *) USBD_VCP_INTERFACE_HS_STRING, USBD_VCP_StrDesc,
                   length);
  }
  else
  {
    USBD_GetString((uint8_t *) USBD_VCP_INTERFACE_FS_STRING, USBD_VCP_StrDesc,
                   length);
  }
  return USBD_VCP_StrDesc;
}
